'use client';

import React, { useContext } from 'react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';

import MenuContext from '@/context/MenuContext/MenuContext';
import MainMenu from '@/sections/MainMenu';
import ContactMenuContent from '@/contents/ContactMenuContent';
import LanguageMenuContent from '@/contents/LanguageMenuContent';
import RatingMenuContent from '@/contents/RatingMenuContent';
import SettingsMenuContent from '@/contents/SettingsMenuContent';
import StartMenuContent from '@/contents/StartMenuContent';
import TutorialMenuContent from '@/contents/TutorialMenuContent';

import ArrowBackIcon from '@/svgs/ArrowBackIcon';
import CloseButtonIcon from '@/svgs/CloseButtonIcon';
import MenuButtonIcon from '@/svgs/MenuButtonIcon';
import Logo from '@/svgs/Logo';
import { checkUrlByLanguage } from '@/utils/checkUrlByLanguage';
import { HeaderContentType } from '@/templates/PageTemplate';
import { Locale, defaultLocale } from '@/i18n.config';

import styles from './Header.module.scss';

export default function Header({
  content, lang = defaultLocale, title = '',
}: {
  content: HeaderContentType; lang: Locale; title?: string;
}) {
  const router = useRouter();

  const { menuStep, handleSetDefault, handleSetMainMenu } = useContext(MenuContext);

  return (
    <header className={styles.Header}>
      <div className={styles[`container${!!title?.length ? '--with-title' : ''}`]}>
        {!title?.length ? (
          <Link
            className={styles.logo}
            href={checkUrlByLanguage(lang, '/')}
            aria-label={content?.logoAriaLabel}
          >
            <Logo />
          </Link>
        ) : (
          <>
            <h1 className={styles.title}>{title}</h1>
            <button className={styles.back} type="button" onClick={() => router.back()}>
              <ArrowBackIcon />
            </button>
          </>
        )}
        <button
          className={styles['toggle-menu']}
          type="button"
          aria-label={content?.menuAriaLabel}
          onClick={menuStep === 'default' ? handleSetMainMenu : handleSetDefault}
        >
          {menuStep === 'default' ? <MenuButtonIcon /> : <CloseButtonIcon />}
        </button>
        <div className={styles[`menu--${menuStep === 'main-menu' ? 'visible' : 'hidden'}`]}>
          <MainMenu content={content?.mainMenu} />
        </div>
        <div className={styles[`menu--${menuStep === 'languages' ? 'visible' : 'hidden'}`]}>
          <LanguageMenuContent
            lang={lang}
            title={content?.languageMenu?.title}
            description={content?.languageMenu?.description}
          />
        </div>
        <div className={styles[`menu--${menuStep === 'settings' ? 'visible' : 'hidden'}`]}>
          <SettingsMenuContent content={content?.settingsMenu} />
        </div>
        <div className={styles[`menu--${menuStep === 'rating' ? 'visible' : 'hidden'}`]}>
          <RatingMenuContent content={content?.ratingMenu} />
        </div>
        <div className={styles[`menu--${menuStep === 'contact' ? 'visible' : 'hidden'}`]}>
          <ContactMenuContent content={content?.contactMenu} />
        </div>
        <div className={styles[`menu--${menuStep === 'tutorial' ? 'visible' : 'hidden'}`]}>
          <TutorialMenuContent content={content?.tutorialMenu} />
        </div>
        <div className={styles[`menu--${menuStep === 'start' ? 'visible' : 'hidden'}`]}>
          <StartMenuContent content={content?.startMenu} />
        </div>
      </div>
    </header>
  );
};
